<?php
// -------------------------------------------------------------------------
// Copyright (c) 3D3.COM Pty Ltd
// All rights reserved
//
// Example JSON data:
//{
// 	"AAA111":
// 	{
// 	   "controltype":"deny",
// 	   "countries":"AU,DE,"
// 	},
// 	"AAA222":
// 	{
// 	   "controltype":"allow",
// 	   "countries":"AU,DE,NL,UK,"
// 	}
//}
// For product code AAA111, if the country is AU or DE, it will be NOT ALLOWED.
// Any other country will be allowed. Statuscode returned will be 0 (SC_OK)
// For product code AAA222, if the country is AU, DE, NL or UK, it will be ALLOWED.
// Any other country will be not allowed. Statuscode returned will be 0 (SC_OK)
// For product codes that aren't configured, it will be ALLOWED for any country.
//  Statuscode returned will be -3 (SC_CODE_NOT_FOUND)
//
//
// *** Query if product code(s) are allowed to be shipped to a country:
//
// ?prdc=QW12~|AAA111~|AAA222&ctry=AU
//
// Response
// { 'QW12':{ 'allow':true, 'statuscode':-3 }, 'AAA111':{ 'allow':false, 'statuscode':0 },'AAA222':{ 'allow':true, 'statuscode':0 } }
//
//
// *** Query to get entire JSON data
//
// ?getcfg
//
// Response
// { 'errmsg':'a message string','statuscode':-1,'data':{ ... } }
//
// The errmsg is only for developers and documentation purposes. It is not
// translated.
//
//
// *** Post to update the JSON file
//
// putcfg={ "AAA111": { "controltype":"deny", "countries":"AU,DE," }, "AAA222": { "controltype":"allow", "countries":"AU,DE,NL,UK," } }
//
// Response
// { 'errmsg':'a message string','statuscode':-1,'data':{ ... } }
//
// The errmsg is only for developers and documentation purposes. It is not
// translated.
// -------------------------------------------------------------------------

error_reporting(E_ALL & ~E_WARNING & ~E_NOTICE & ~E_STRICT);

// Define response status codes
define("SC_OK", 0);
define("SC_INVALID_QUERY_STRING", -1);
define("SC_INVALID_DATA_FILE", -2);
define("SC_CODE_NOT_FOUND", -3);
define("SC_CANT_LOCK_DATA_FILE", -4);
define("SC_FAILED_TO_READ_FROM_DATA_FILE", -5);
define("SC_FAILED_TO_WRITE_TO_DATA_FILE", -6);
define("SC_NO_DATA", -9);

define('OPENLOCK_TYPE_NONE', -1);
define('OPENLOCK_TYPE_FLOCK', 0);
define('OPENLOCK_TYPE_SEM', 1);

$response = array();
$cfg = array();
$cfgfile = 'prodCtryShip.json';

// Handle update requests from GlobeCharge
if (isset($_POST['putcfg'])) {
	// Open for reading and truncating
	$olock = new openlock($cfgfile, "r+");
	$fp = $olock->get_fp();
	// Ensure the file handle is valid and lock it for exclusive access
	// (it will wait if the file has already been locked)
	if ($fp && $olock->lock()) {
		ftruncate($fp, 0);
		fseek($fp, 0);

		$response['statuscode'] = SC_OK;
		if (!fwrite($fp, $_REQUEST['putcfg'])) {
			$response['errmsg'] = 'Error: failed to write to data file';
			$response['statuscode'] = SC_FAILED_TO_WRITE_TO_DATA_FILE;
		}

		$olock->unlock();
		$olock->close();
	} else {
		$response['errmsg'] = 'Error: can\'t lock data file';
		$response['statuscode'] = SC_CANT_LOCK_DATA_FILE;
	}
} else if (isset($_REQUEST['getcfg'])) {
	try {
		$cfgdata = file_get_contents($cfgfile);
		$response['data'] = $cfgdata;
		$response['statuscode'] = SC_OK;
	} catch (Exception $e) {
		$response['errmsg'] = 'Error: no data';
		$response['statuscode'] = SC_NO_DATA;
	}
} else if (isset($_REQUEST['prdc']) && isset($_REQUEST['ctry'])) {
	$prdcReq = $_REQUEST['prdc'];
	$ctryReq = $_REQUEST['ctry'];

	try {
		$cfgdata = file_get_contents($cfgfile);
		$cfg = json_decode($cfgdata, true);
		if ($cfg) {
			$prdcList = explode('~|', $prdcReq);
			for ($i = 0, $prdcNum = count($prdcList); $i < $prdcNum; ++$i) {
				$prdc = $prdcList[$i];
				if ($cfg[$prdc]) {
					$isCtryInList = (strpos($cfg[$prdc]['countries'], $ctryReq.',') !== false);
					if ($cfg[$prdc]['controltype'] == 'deny') {
						$response[$prdc]['allow'] = !$isCtryInList;
						$response[$prdc]['statuscode'] = SC_OK;
					}
					else if ($cfg[$prdc]['controltype'] == 'allow') {
						$response[$prdc]['allow'] = $isCtryInList;
						$response[$prdc]['statuscode'] = SC_OK;
					}
				} else {
					$response[$prdc]['allow'] = true;
					$response[$prdc]['statuscode'] = SC_CODE_NOT_FOUND;
				}
			}
		} else {
			$response['errmsg'] = 'Error: invalid data file';
			$response['statuscode'] = SC_INVALID_DATA_FILE;
		}
	} catch (Exception $e) {
		$response['errmsg'] = 'Error: failed to read from data file';
		$response['statuscode'] = SC_FAILED_TO_READ_FROM_DATA_FILE;
	}
} else {
	$response['errmsg'] = 'Error: invalid query string';
	$response['statuscode'] = SC_INVALID_QUERY_STRING;
}


//--------------------------------------------------------------------------
// Utility functions
//--------------------------------------------------------------------------

// No ftok on Windows...
if (!function_exists('ftok')) {
	function ftok($filename = "", $proj = "")
	{
		if (empty($filename) || !file_exists($filename)) {
			return -1;
		} else {
			$filename = $filename . (string) $proj;
			for ($key = array(); sizeof($key) < strlen($filename); $key[] = ord(substr($filename, sizeof($key), 1)));
			return dechex(array_sum($key));
		}
	}
}

class openlock
{
	/*private */
	var $lock_type = OPENLOCK_TYPE_NONE;
	/*private */
	var $fp = null;
	/*private */
	var $filename;

	/*private */
	var $sem_handle;

	function openlock($filename, $mode)
	{
		$this->filename = $filename;
		$this->fp = fopen($this->filename, $mode);
	}

	function close()
	{
		$this->unlock();
		fclose($this->fp);
	}

	/*public */
	function get_fp()
	{
		return $this->fp;
	}

	/*public */
	function lock()
	{
		if ($this->fp != null && $this->lock_type == OPENLOCK_TYPE_NONE) {
			if (get_resource_type($this->fp) == "file") {
				if ($this->f_lock()) {
					$this->lock_type = OPENLOCK_TYPE_FLOCK;
					return true;
				}
			}

			if ($this->sem_lock()) {
				$this->lock_type = OPENLOCK_TYPE_SEM;
				return true;
			} else if ($this->f_lock()) {
				$this->lock_type = OPENLOCK_TYPE_FLOCK;
				return true;
			} else {
				return false;
			}
		}

		return false;
	}

	/*public */
	function unlock()
	{
		if ($this->lock_type != OPENLOCK_TYPE_NONE) {
			switch ($this->lock_type) {
				case OPENLOCK_TYPE_FLOCK:
					$this->f_unlock();
					$this->lock_type = OPENLOCK_TYPE_NONE;
					break;
				case OPENLOCK_TYPE_SEM:
					$this->sem_unlock();
					$this->lock_type = OPENLOCK_TYPE_NONE;
					break;
			}
		}
	}

	// Works on non-NFSed Windows and Unix
	/*private */
	function f_lock()
	{
		return flock($this->fp, LOCK_EX);
	}

	/*private */
	function f_unlock()
	{
		return flock($this->fp, LOCK_UN);
	}

	// Works on Unix
	/*private */
	function sem_lock()
	{
		if (function_exists('ftok') && function_exists('sem_get') && function_exists('sem_acquire')) {
			$sem_key = ftok($this->filename, 'R');
			if ($this->sem_handle = sem_get($sem_key, 1)) {
				if (sem_acquire($this->sem_handle)) {
					return true;
				}
			}
		}
		return false;
	}

	/*private */
	function sem_unlock()
	{
		return sem_release($this->sem_handle);
	}
}

//--------------------------------------------------------------------------
// Return response and data
//--------------------------------------------------------------------------

// Output in JSON format
$output = json_encode($response);
header('Content-Type: application/json');
print($output);
